#version 400 compatibility

/*
====================================================================================================

    Copyright (C) 2020 RRe36

    All Rights Reserved unless otherwise explicitly stated.


    By downloading this you have agreed to the license and terms of use.
    These can be found inside the included license-file
    or here: https://rre36.com/copyright-license

    Violating these terms may be penalized with actions according to the Digital Millennium
    Copyright Act (DMCA), the Information Society Directive and/or similar laws
    depending on your country.

====================================================================================================
*/

/*DRAWBUFFERS:0*/
layout(location = 0) out vec4 scene;

#include "/lib/head.glsl"
#include "/lib/util/encoders.glsl"
#include "/lib/util/colorspace.glsl"

const int shadowMapResolution   = 2560;     //[512 1024 1536 2048 2560 3072 3584 4096 6144 8192 16384]
const float shadowDistance      = 128.0;

const bool shadowHardwareFiltering = true;

in vec2 coord;

flat in vec3 cloudLightColor;

flat in mat4x3 lightColor;
flat in mat2x3 skyColorMat;

uniform sampler2D colortex0;
uniform sampler2D colortex1;
uniform sampler2D colortex2;
uniform sampler2D colortex3;
uniform sampler2D colortex5;
uniform sampler2D colortex7;

uniform sampler2D depthtex0;
uniform sampler2D depthtex1;

uniform sampler2D noisetex;

uniform sampler2DShadow shadowtex0;
uniform sampler2DShadow shadowtex1;
uniform sampler2D shadowcolor0;

uniform int frameCounter;
uniform int isEyeInWater;
uniform int worldTime;

uniform float eyeAltitude;
uniform float far;
uniform float frameTimeCounter;
uniform float lightFlip;
uniform float sunAngle;
uniform float cloudLightFlip;
uniform float worldAnimTime;

uniform ivec2 eyeBrightness;
uniform ivec2 eyeBrightnessSmooth;

uniform vec2 taaOffset;
uniform vec2 viewSize;
uniform vec2 skyCaptureResolution;

uniform vec3 cameraPosition;
uniform vec3 upvec, upvecView;
uniform vec3 sunvec, sunvecView;
uniform vec3 moonvec, moonvecView;
uniform vec3 lightvec, lightvecView;
uniform vec3 cloudLightDir, cloudLightDirView;

uniform vec4 daytime;

uniform mat4 gbufferModelView, gbufferModelViewInverse;
uniform mat4 gbufferProjection, gbufferProjectionInverse;
uniform mat4 shadowModelView, shadowModelViewInverse;
uniform mat4 shadowProjection, shadowProjectionInverse;


/* ------ includes ------ */
#define FUTIL_TBLEND
#define FUTIL_MAT16
#include "/lib/fUtil.glsl"

#include "/lib/frag/bluenoise.glsl"
#include "/lib/util/transforms.glsl"
#include "/lib/atmos/phase.glsl"
#include "/lib/light/warp.glsl"
#include "/lib/frag/noise.glsl"

#include "/lib/atmos/skyGradient.glsl"

#include "/lib/atmos/waterConst.glsl"

vec3 simpleFog(vec3 scene, float d, float cave) {
    float density   = d * 1e-5 * fogDensityMult;

    float transmittance = expf(-density);

    vec3 light      = (sunAngle<0.5 ? lightColor[0] : lightColor[2]) * lightFlip * 0.03 * sqr(cave) + lightColor[1] * cave;

    return scene * transmittance + light * density * pi;
}
vec3 waterFog(vec3 scene, float d, vec3 color) {
    float density   = max(0.0, d) * waterDensity;

    vec3 transmittance = expf(-waterAttenCoeff * density);

    const vec3 scatterCoeff = vec3(5e-2, 1e-1, 2e-1);

    vec3 scatter    = 1.0-exp(-density * scatterCoeff);
        scatter    *= max(expf(-waterAttenCoeff * density), expf(-waterAttenCoeff * tau));

    return scene * transmittance + scatter * color * rcp(pi);
}
vec3 lavaFog(vec3 scene, float d) {
    float density   = max(0.0, d);

    float transmittance = expf(-1.0 * density);

    return mix(vec3(1.0, 0.3, 0.02), scene, transmittance);
}

vec3 chunkEdgeFog(vec3 scene, float sceneDirY, vec3 skyColor, float distance) {
    return mix(scene, skyColor, sstep(distance, far * 0.6, far) * 0.66 * sstep(-sceneDirY, -0.2, 0.0));
}

vec3 getShadowmapPos(vec3 scenePos, const float bias) {  //shadow 2d
    vec3 pos    = scenePos + vec3(bias) * lightvec;
    float a     = length(pos);
        pos     = viewMAD(shadowModelView, pos);
        pos     = projMAD(shadowProjection, pos);
        pos.z  *= 0.2;
        pos.z  -= 0.0012*(saturate(a/256.0));

        pos.xy  = shadowmapWarp(pos.xy);

    return pos*0.5+0.5;
}
vec3 toShadowmapTexturePos(vec3 shadowPos) {
    //shadowPos.z    *= 0.2;
    shadowPos.xy    = shadowmapWarp(shadowPos.xy) * 0.5 + 0.5;

    return shadowPos;
}

vec3 getShadowcol(sampler2D tex, vec2 coord) {
    vec4 x  = texture(tex, coord);
        x.rgb = sqr(x.rgb);

    if (x.a < 0.97 && x.a > 0.93) {
        x.rgb = vec3(0.1, 0.2, 1.0);
    }

    return mix(vec3(1.0), x.rgb, x.a);
}

/* ------ volumetric fog ------ */

const float airMieG   = 0.7;

const vec3 airRayleighCoeff     = vec3(3.54e-6, 5.47e-6, 7.62e-6);
const vec3 airMieCoeff          = vec3(9e-6);    //19e-5 for rain

const mat2x3 airScatterMat      = mat2x3(airRayleighCoeff, airMieCoeff);
const mat2x3 airExtinctMat      = mat2x3(airRayleighCoeff, airMieCoeff);

const vec3 fogMistCoeff         = vec3(4e-3);
const mat3x3 fogScatterMat      = mat3x3(airRayleighCoeff, airMieCoeff, fogMistCoeff);
const mat3x3 fogExtinctMat      = mat3x3(airRayleighCoeff, airMieCoeff * 1.1, fogMistCoeff);

const vec2 fogFalloffScale  = 1.0 / vec2(4e1, 25e1);
const vec2 fogAirScale      = vec2(100.0, 1e2);

vec2 airPhaseFunction(float cosTheta) {
    return vec2(rayleighPhase(cosTheta), mieCS(cosTheta, airMieG));
}
float airMieBackscatter(float cosTheta) {
    return mieHG(cosTheta, -airMieG * rcp(pi));
}
float fogMistPhase(float cosTheta, float density) {
    return mix(mieCS(cosTheta, pow(airMieG, 1.0 + density)), airMieBackscatter(cosTheta), 0.19);
}

uniform vec3 fogDensityCoeff;

#define fogMistAltitude 100.0
#define airDensity 1.0

#ifdef volumeWorldTimeAnim
    float fogTime     = worldAnimTime * 3.6;
#else
    float fogTime     = frameTimeCounter * 0.006;
#endif

float mistDensity(vec3 rPos, float altitude) {
    vec3 wind   = vec3(fogTime, 0.0, fogTime * 0.5);
    float noise = value3D(rPos * 0.037 + wind) + value3D(rPos * 0.08 + wind * 3.0) * 0.5;

    float mist  = expf(-max0((altitude - 64.0) * 0.01));
        mist    = sstep(noise * rcp(1.5) - (1.0 - mist) * 0.5, 0.33, 0.6);
        mist   *= sqr(1.0 - linStep(altitude, 85.0, fogMistAltitude));

    return mist * fogDensityCoeff.z;
}

vec3 fogAirDensity(vec3 rPos, float altitude, bool terrain) {
    //float dist  = length(rPos);
    rPos   += cameraPosition;

    float maxFade = sqr(1.0 - linStep(altitude, 144.0, 256.0));

    vec2 rm     = expf(-max0((altitude - 66.0) * fogFalloffScale));
    //rm.y *= 0;
    float mist  = 0.0;
    if (fogDensityCoeff.z > 1e-5) mist = mistDensity(rPos, altitude);
    //if (terrain) rm *= 1.0 + sstep(dist, far * 0.6, far) * pi * 50.0;

    return vec3(rm * fogAirScale * fogDensityCoeff.xy * maxFade * fogDensityMult, mist);
}

float fogMistLightOD(vec3 pos) {
    float stepsize = 12.0;

    const int steps = 2;

    float od = 0.0;

    for(uint i = 0; i < steps; ++i, pos += lightvec * stepsize) {

        if(pos.y > fogMistAltitude || pos.y < 0.0) continue;
        
        float density = mistDensity(pos, pos.y);
        if (density <= 0.0) continue;

            od += density * stepsize;
    }

    return od;
}

mat2x3 volumetricFog(vec3 scenePos, vec3 sceneDir, bool isSky, float vDotL, float dither, float cave) {
    float topDist    = length(sceneDir * ((256.0 - eyeAltitude) * rcp(sceneDir.y)));
    float bottomDist = length(sceneDir * ((-32.0 - eyeAltitude) * rcp(sceneDir.y)));

    float volumeHull = sceneDir.y > 0.0 ? topDist : bottomDist;

    float endDist   = isSky ? min(volumeHull, fogClipDist) : length(scenePos);
    float startDist = eyeAltitude > 256.0 ? topDist : 1.0;

    vec3 startPos   = eyeAltitude > 256.0 ? sceneDir * startDist : vec3(0.0);
        startPos   += gbufferModelViewInverse[3].xyz;
    vec3 endPos     = isSky ? sceneDir * endDist : scenePos;

    float baseStep  = length(endPos - startPos);
    float stepCoeff = saturate(baseStep * rcp(clamp(far, 256.0, 512.0)));

    uint steps      = fogMinSteps + uint(stepCoeff * fogAdaptiveSteps);

    vec3 rStep      = (endPos - startPos) / float(steps);
    vec3 rPos       = startPos + rStep * dither;
    float rLength   = length(rStep);

    vec3 shadowStartPos = viewMAD(shadowModelView, (startPos));
        shadowStartPos  = projMAD(shadowProjection, shadowStartPos);
        shadowStartPos.z *= 0.2;
    vec3 shadowEndPos   = viewMAD(shadowModelView, (endPos));
        shadowEndPos    = projMAD(shadowProjection, shadowEndPos);
        shadowEndPos.z *= 0.2;

    vec3 shadowStep = (shadowEndPos - shadowStartPos) / float(steps);
    vec3 shadowPos  = shadowStartPos + shadowStep * dither;

    mat2x3 scattering = mat2x3(0.0);
    vec3 transmittance = vec3(1.0);

    vec3 phase;
        phase.xy    = airPhaseFunction(vDotL);
        phase.z     = mix(mieCS(vDotL, airMieG * mistMieAnisotropy), airMieBackscatter(vDotL), mistMieAnisotropy / 4.0);
    float phaseIso  = 0.25;

    vec3 sunlight       = (worldTime>23000 || worldTime<12900) ? lightColor[0] : lightColor[2];
        sunlight       *= lightFlip * sqrt2;
    vec3 skylight       = lightColor[1] * cave;

    #ifdef lightleakWorkaroundToggle
        sunlight       *= cave;
    #endif

    float pFade         = saturate(mieHG(vDotL, 0.65));

    for (uint i = 0; i < steps; ++i, rPos += rStep, shadowPos += shadowStep) {
        if (max3(transmittance) < 0.01) break;

        float altitude  = rPos.y + eyeAltitude;

        if (altitude > 256.0) continue;

        vec3 density    = fogAirDensity(rPos, altitude, !isSky);

        //if (max3(density) < 1e-32) continue;

        vec3 stepRho    = density * rLength;
        vec3 od         = fogExtinctMat * stepRho;

        vec3 stepT      = expf(-od);
        vec3 scatterInt = saturate((stepT - 1.0) * rcp(-max(od, 1e-16)));
        vec3 visScatter = transmittance * scatterInt;

        #ifdef fogMistAdvanced
            if (density.z > 0.0) {
                float mistLightOD = fogMistLightOD(rPos + cameraPosition);

                float mistLighting  = expf(-stepRho.z) * pow(1.0 + 1.0 * 0.7 * mistLightOD, -1.0 / 0.7);
                    mistLighting    = mix(pi * (1.0 - 0.6 * expf(-density.z * 28.0)), 1.0, pFade);
                    mistLighting    = fogMistPhase(vDotL, mistLightOD);
                phase.z     = mistLighting;
            }
        #else
            if (density.z > 0.0) {
                phase.z     = fogMistPhase(vDotL, density.z) * (2.0 - pFade);
            }
        #endif

        vec3 sunScatter = fogScatterMat * (stepRho * phase) * visScatter;
        vec3 skyScatter = fogScatterMat * (stepRho * phaseIso) * visScatter;

        vec3 shadowCoord = vec3(shadowmapWarp(shadowPos.xy), shadowPos.z) * 0.5 + 0.5;

        float shadow0   = texture(shadowtex0, shadowCoord);

        float shadow    = 1.0;
        vec3 shadowCol  = vec3(1.0);
        
        if (shadow0 < 1.0) {
            shadow      = texture(shadowtex1, shadowCoord);

            if (abs(shadow - shadow0) > 0.1) {
                shadowCol   = getShadowcol(shadowcolor0, shadowCoord.xy);
            }
        }

        scattering[0]  += (sunScatter * shadowCol * transmittance) * shadow;
        scattering[1]  += skyScatter * transmittance;

        transmittance  *= stepT;
    }

    vec3 color  = scattering[0] * sunlight + scattering[1] * skylight;

    if (color != color) {   //because NaNs on nVidia don't need a logic cause to happen
        color = vec3(0.0);
        transmittance = vec3(1.0);
    }

    return mat2x3(color, transmittance);
}

mat2x3 volumetricFog2(vec3 scenePos, vec3 startPos0, vec3 sceneDir, bool isSky, float vDotL, float dither, float cave) {
    float topDist    = length(sceneDir * ((256.0 - eyeAltitude) * rcp(sceneDir.y)));
    float bottomDist = length(sceneDir * ((-32.0 - eyeAltitude) * rcp(sceneDir.y)));

    float volumeHull = sceneDir.y > 0.0 ? topDist : bottomDist;

    float endDist   = isSky ? min(volumeHull, fogClipDist) : length(scenePos);
    //float startDist = eyeAltitude > 256.0 ? topDist : 1.0;

    vec3 startPos   = startPos0;
    vec3 endPos     = isSky ? sceneDir * endDist : scenePos;

    float baseStep  = length(endPos - startPos);
    float stepCoeff = saturate(baseStep * rcp(clamp(far, 256.0, 512.0)));

    uint steps      = 6 + uint(stepCoeff * 8);

    vec3 rStep      = (endPos - startPos) / float(steps);
    vec3 rPos       = startPos + rStep * dither;
    float rLength   = length(rStep);

    vec3 shadowStartPos = viewMAD(shadowModelView, (startPos));
        shadowStartPos  = projMAD(shadowProjection, shadowStartPos);
        shadowStartPos.z *= 0.2;
    vec3 shadowEndPos   = viewMAD(shadowModelView, (endPos));
        shadowEndPos    = projMAD(shadowProjection, shadowEndPos);
        shadowEndPos.z *= 0.2;

    vec3 shadowStep = (shadowEndPos - shadowStartPos) / float(steps);
    vec3 shadowPos  = shadowStartPos + shadowStep * dither;

    mat2x3 scattering = mat2x3(0.0);
    vec3 transmittance = vec3(1.0);

    vec3 phase;
        phase.xy    = airPhaseFunction(vDotL);
        phase.z     = mix(mieCS(vDotL, airMieG * mistMieAnisotropy), airMieBackscatter(vDotL), mistMieAnisotropy / 4.0);
    float phaseIso  = 0.25;

    vec3 sunlight       = (worldTime>23000 || worldTime<12900) ? lightColor[0] : lightColor[2];
        sunlight       *= lightFlip * sqrt2;
    vec3 skylight       = lightColor[1] * cave;

    #ifdef lightleakWorkaroundToggle
        sunlight       *= cave;
    #endif

    float pFade         = saturate(mieHG(vDotL, 0.65));

    for (uint i = 0; i < steps; ++i, rPos += rStep, shadowPos += shadowStep) {
        if (max3(transmittance) < 0.01) break;

        float altitude  = rPos.y + eyeAltitude;

        if (altitude > 256.0) continue;

        vec3 density    = fogAirDensity(rPos, altitude, !isSky);

        //if (max3(density) < 1e-32) continue;

        vec3 stepRho    = density * rLength;
        vec3 od         = fogExtinctMat * stepRho;

        vec3 stepT      = expf(-od);
        vec3 scatterInt = saturate((stepT - 1.0) * rcp(-max(od, 1e-16)));
        vec3 visScatter = transmittance * scatterInt;

        #ifdef fogMistAdvanced
            if (density.z > 0.0) {
                float mistLightOD = fogMistLightOD(rPos + cameraPosition);

                float mistLighting  = expf(-stepRho.z) * pow(1.0 + 1.0 * 0.7 * mistLightOD, -1.0 / 0.7);
                    mistLighting    = mix(pi * (1.0 - 0.6 * expf(-density.z * 28.0)), 1.0, pFade);
                    mistLighting    = fogMistPhase(vDotL, mistLightOD);
                phase.z     = mistLighting;
            }
        #else
            if (density.z > 0.0) {
                phase.z     = fogMistPhase(vDotL, density.z) * (2.0 - pFade);
            }
        #endif

        vec3 sunScatter = fogScatterMat * (stepRho * phase) * visScatter;
        vec3 skyScatter = fogScatterMat * (stepRho * phaseIso) * visScatter;

        vec3 shadowCoord = vec3(shadowmapWarp(shadowPos.xy), shadowPos.z) * 0.5 + 0.5;

        float shadow0   = texture(shadowtex0, shadowCoord);

        float shadow    = 1.0;
        vec3 shadowCol  = vec3(1.0);
        
        if (shadow0 < 1.0) {
            shadow      = texture(shadowtex1, shadowCoord);

            if (abs(shadow - shadow0) > 0.1) {
                shadowCol   = getShadowcol(shadowcolor0, shadowCoord.xy);
            }
        }

        scattering[0]  += (sunScatter * shadowCol * transmittance) * shadow;
        scattering[1]  += skyScatter * transmittance;

        transmittance  *= stepT;
    }

    vec3 color  = scattering[0] * sunlight + scattering[1] * skylight;

    if (color != color) {   //because NaNs on nVidia don't need a logic cause to happen
        color = vec3(0.0);
        transmittance = vec3(1.0);
    }

    return mat2x3(color, transmittance);
}

void applyFogData(inout vec3 color, in mat2x3 data) {
    color = color * data[1] + data[0];
}


/* ------ refraction ------ */

vec3 refract2(vec3 I, vec3 N, vec3 NF, float eta) {     //from spectrum by zombye
    float NoI = dot(N, I);
    float k = 1.0 - eta * eta * (1.0 - NoI * NoI);
    if (k < 0.0) {
        return vec3(0.0); // Total Internal Reflection
    } else {
        float sqrtk = sqrt(k);
        vec3 R = (eta * dot(NF, I) + sqrtk) * NF - (eta * NoI + sqrtk) * N;
        return normalize(R * sqrt(abs(NoI)) + eta * I);
    }
}


#include "/lib/frag/gradnoise.glsl"

#include "/lib/atmos/clouds.glsl"

vec4 cloudSystem(vec3 worldDir, vec3 worldPos, bool terrain, float vDotL, float dither, vec3 skyColor) {
    vec3 totalScattering    = vec3(0.0);
    float totalTransmittance = 1.0;

    vec3 sunlight       = (worldTime>23000 || worldTime<12900) ? cloudLightColor : lightColor[2];
        sunlight       /= sqrt3;
        sunlight       *= cloudLightFlip;
    vec3 skylight       = mix(skyColorMat[0], lightColor[1] * (1.0 - sqrt(daytime.w)*0.96), 0.1) / pi;

    float pFade         = saturate(mieHG(vDotL, 0.65));

    float within    = sstep(eyeAltitude, float(cloudVolume0Alt) - 15.0, float(cloudVolume0Alt)) * (1.0 - sstep(eyeAltitude, cloudVol0MaxY, cloudVol0MaxY + 15.0));
    bool isBelowVol = eyeAltitude < cloudVol0MidY;
    bool visibleVol = worldDir.y > 0.0 && isBelowVol || worldDir.y < 0.0 && !isBelowVol;

    float lightNoise = ditherGradNoiseTemporal();

    #ifdef cloudVolumeStoryMode
        const float sigmaA  = 0.1;
            sunlight       *= pi * sqrt2;
            skylight       /= sqrt2;
        const float sigmaT  = 1.25;
    #else
        const float sigmaA  = 0.2;
        const float sigmaT  = 1.125;
    #endif

    if (visibleVol || within > 0.0) {
        vec3 bottom     = worldDir * ((cloudVolume0Alt - eyeAltitude) * rcp(worldDir.y));
        vec3 top        = worldDir * ((cloudVol0MaxY - eyeAltitude) * rcp(worldDir.y));

        if (worldDir.y < 0.0 && isBelowVol || worldDir.y > 0.0 && !isBelowVol) {
            bottom      = vec3(0.0);
            top         = vec3(0.0);
        }

        vec3 start      = isBelowVol ? bottom : top;
        vec3 end        = isBelowVol ? top : bottom;
            start       = mix(start, gbufferModelViewInverse[3].xyz, within);
            end         = mix(end, worldDir * cloudVolume0Clip, within);

        if (terrain) end = worldPos;

        float stepCoeff     = 1.0 + sqr(max(within, float(terrain))) * 4.0;
        uint steps          = uint(cloudVolume0Samples * stepCoeff);

        vec3 rStep          = (end - start) * rcp(float(steps));
        vec3 rPos           = rStep * dither + start + cameraPosition;
        float rLength       = length(rStep);

        vec3 scattering     = vec3(0.0);
        float transmittance = 1.0;

        for (uint i = 0; i < steps; ++i, rPos += rStep) {
            if (transmittance < 0.01) break;
            if (rPos.y < cloudVolume0Alt || rPos.y > cloudVol0MaxY) continue;

            float dist  = distance(rPos, cameraPosition);
            if (dist > cloudVolume0Clip) continue;

            float density = cloudVolume0Shape(rPos);
            if (density <= 0.0) continue;

            float extinction    = density * sigmaT;
            float stepT         = exp(-extinction * rLength);
            float integral      = (1.0 - stepT) * rcp(sigmaT);

            vec3 stepScatter    = vec3(0.0);

            float lightOD       = cloudVolume0LightOD(rPos, 5, cloudLightDir) * sigmaA;
            float skyOD         = cloudVolume0LightOD(rPos, 4, vec3(0.0, 1.0, 0.0)) * sigmaA;

            float powder        = 8.0 * (1.0 - 0.97 * exp(-extinction * 18.0));

            #ifdef cloudVolumeStoryMode
                float anisoPowder   = 1.0;
            #else
                float anisoPowder   = mix(powder, 1.0, pFade);
            #endif

            vec3 phaseG         = pow(vec3(0.45, 0.25, 0.95), vec3(1.0 + lightOD));

            float phase = cloudPhase(vDotL, 1.0, phaseG);

            stepScatter.x  += max(expf(-lightOD * sigmaT), expf(-lightOD * sigmaT * 0.2) * 0.75) * phase * anisoPowder * sigmaT;
            stepScatter.y  += max(expf(-skyOD * sigmaT), expf(-skyOD * sigmaT * 0.2) * 0.75) * powder * sigmaT;

            stepScatter     = (sunlight * stepScatter.x) + (skylight * stepScatter.y);

            float atmosFade = expf(-dist * 2.8e-3);

            stepScatter     = mix(skyColor * sigmaT, stepScatter, atmosFade);

            scattering     += stepScatter * (integral * transmittance);

            transmittance  *= stepT;
        }

        transmittance       = linStep(transmittance, 0.01, 1.0);

        totalScattering    += scattering;
        totalTransmittance *= transmittance;
    }

    #ifdef cloudVolume1Enabled
    within    = sstep(eyeAltitude, float(cloudVolume1Alt) - 15.0, float(cloudVolume1Alt)) * (1.0 - sstep(eyeAltitude, cloudVol1MaxY, cloudVol1MaxY + 15.0));
    isBelowVol = eyeAltitude < cloudVol1MidY;
    visibleVol = worldDir.y > 0.0 && isBelowVol || worldDir.y < 0.0 && !isBelowVol;

    if (visibleVol || within > 0.0) {
        vec3 bottom     = worldDir * ((cloudVolume1Alt - eyeAltitude) * rcp(worldDir.y));
        vec3 top        = worldDir * ((cloudVol1MaxY - eyeAltitude) * rcp(worldDir.y));

        if (worldDir.y < 0.0 && isBelowVol || worldDir.y > 0.0 && !isBelowVol) {
            bottom      = vec3(0.0);
            top         = vec3(0.0);
        }

        vec3 start      = isBelowVol ? bottom : top;
        vec3 end        = isBelowVol ? top : bottom;
            start       = mix(start, gbufferModelViewInverse[3].xyz, within);
            end         = mix(end, worldDir * cloudVolume0Clip, within);

        if (terrain) end = worldPos;

        float stepCoeff     = 1.0 + sqr(within) * 4.0;
        uint steps          = uint(cloudVolume1Samples * stepCoeff);

        vec3 rStep          = (end - start) * rcp(float(steps));
        vec3 rPos           = rStep * dither + start + cameraPosition;
        float rLength       = length(rStep);

        vec3 scattering     = vec3(0.0);
        float transmittance = 1.0;

        for (uint i = 0; i < steps; ++i, rPos += rStep) {
            if (transmittance < 0.01) break;
            if (rPos.y < cloudVolume1Alt || rPos.y > cloudVol1MaxY) continue;

            float dist  = distance(rPos, cameraPosition);
            if (dist > cloudVolume1Clip) continue;

            float density = cloudVolume1Shape(rPos);
            if (density <= 0.0) continue;

            float extinction    = density * sigmaT;
            float stepT         = exp(-extinction * rLength);
            float integral      = (1.0 - stepT) * rcp(sigmaT);

            vec3 stepScatter    = vec3(0.0);

            float lightOD       = cloudVolume1LightOD(rPos, 4, cloudLightDir) * sigmaA;
            float skyOD         = cloudVolume1LightOD(rPos, 4, vec3(0.0, 1.0, 0.0)) * sigmaA;

            float powder        = 8.0 * (1.0 - 0.97 * exp(-extinction * 18.0)); 

            #ifdef cloudVolumeStoryMode
                float anisoPowder   = 1.0;
            #else
                float anisoPowder   = mix(powder, 1.0, pFade);
            #endif

            vec3 phaseG         = pow(vec3(0.45, 0.25, 0.95), vec3(1.0 + lightOD));

            float phase = cloudPhase(vDotL, 1.0, phaseG);

            stepScatter.x  += max(expf(-lightOD * sigmaT), expf(-lightOD * sigmaT * 0.2) * 0.75) * phase * anisoPowder * sigmaT;
            stepScatter.y  += max(expf(-skyOD * sigmaT), expf(-skyOD * sigmaT * 0.2) * 0.75) * powder * sigmaT;

            stepScatter     = (sunlight * stepScatter.x) + (skylight * stepScatter.y);

            float atmosFade = expf(-dist * 2.8e-3 * 0.9);

            stepScatter     = mix(skyColor * sigmaT, stepScatter, atmosFade);

            scattering     += stepScatter * (integral * transmittance);

            transmittance  *= stepT;
        }

        transmittance       = linStep(transmittance, 0.01, 1.0);

        totalScattering    += scattering * totalTransmittance;
        totalTransmittance *= transmittance;
    }
    #endif

    return vec4(totalScattering, totalTransmittance);
}

/* ------ reflections ------ */

#include "/lib/light/brdf.glsl"

#include "/lib/frag/labPBR.glsl"

#include "/lib/frag/ssr.glsl"

/*
These two functions used for rough reflections are based on zombye's spectrum shaders
https://github.com/zombye/spectrum
*/

mat3 getRotationMat(vec3 x, vec3 y) {
	float cosine = dot(x, y);
	vec3 axis = cross(y, x);

	float tmp = 1.0 / dot(axis, axis);
	      tmp = tmp - tmp * cosine;
	vec3 tmpv = axis * tmp;

	return mat3(
		axis.x * tmpv.x + cosine, axis.x * tmpv.y - axis.z, axis.x * tmpv.z + axis.y,
		axis.y * tmpv.x + axis.z, axis.y * tmpv.y + cosine, axis.y * tmpv.z - axis.x,
		axis.z * tmpv.x - axis.y, axis.z * tmpv.y + axis.x, axis.z * tmpv.z + cosine
	);
}
vec3 ggxFacetDist(vec3 viewDir, float roughness, vec2 xy) {
	/*
    GGX VNDF sampling
	http://www.jcgt.org/published/0007/04/01/
    */

    viewDir     = normalize(vec3(roughness * viewDir.xy, viewDir.z));

    float clsq  = dot(viewDir.xy, viewDir.xy);
    vec3 T1     = vec3(clsq > 0.0 ? vec2(-viewDir.y, viewDir.x) * inversesqrt(clsq) : vec2(1.0, 0.0), 0.0);
    vec3 T2     = vec3(-T1.y * viewDir.z, viewDir.z * T1.x, viewDir.x * T1.y - T1.x * viewDir.y);

	float r     = sqrt(xy.x);
	float phi   = tau * xy.y;
	float t1    = r * cos(phi);
	float a     = saturate(1.0 - t1 * t1);
	float t2    = mix(sqrt(a), r * sin(phi), 0.5 + 0.5 * viewDir.z);

	vec3 normalH = t1 * T1 + t2 * T2 + sqrt(saturate(a - t2 * t2)) * viewDir;

	return normalize(vec3(roughness * normalH.xy, normalH.z));
}

vec2 unprojectSphere(vec3 dir) {
    vec2 lonlat     = vec2(atan(-dir.x, dir.z), acos(dir.y));
    return lonlat * vec2(rcp(tau), rcp(pi)) + vec2(0.5, 0.0);
}

void main() {
    vec4 sceneColor   = stex(colortex0);
    //vec4 tex1       = stex(colortex1);
    vec4 tex2       = stex(colortex2);
    vec4 tex3       = stex(colortex3);

    vec4 tex1       = stex(colortex1);
    vec2 sceneLmap  = decode2x8(tex1.z);
    vec3 sceneNormal = decodeNormal(tex1.xy);
    vec3 viewNormal = normalize(mat3(gbufferModelView) * sceneNormal);
    
    //~4fps for the whole part below
    
    int matID      = decodeMatID16(tex2.x);

    float sceneDepth0   = stex(depthtex0).x;
    vec3 viewPos0       = screenToViewSpace(vec3(coord, sceneDepth0));
    vec3 scenePos0      = viewToSceneSpace(viewPos0);
    vec3 sceneDir0      = normalize(scenePos0);

    float sceneDepth1   = stex(depthtex1).x;
    vec3 viewPos1       = screenToViewSpace(vec3(coord, sceneDepth1));
    vec3 scenePos1      = viewToSceneSpace(viewPos1);
    vec3 sceneDir1      = normalize(scenePos1);

    float vDotL     = dot(normalize(viewPos1), lightvecView);
    
    bool translucent = (sceneDepth0<sceneDepth1);

    bool water      = matID == 102;

    vec3 translucentAlbedo = sqr(decode3x8(tex2.z));

    materialProperties material = materialProperties(0.0001, 0.02, false, false, mat2x3(1.0));

    if (!water) material = decodeLabBasic(decode2x8(tex2.y));

    float wetness   = 0.0;

    float caveMult  = linStep(eyeBrightnessSmooth.y/240.0, 0.1, 0.9);

    vec3 fogcol     = (sunAngle<0.5 ? lightColor[0] : lightColor[2]) * lightFlip * mieHG(vDotL, 0.68) * sqr(caveMult);
        fogcol     *= mix(0.33, 0.66, sqrt(abs(lightvec.y)));
        fogcol     += lightColor[1] * caveMult * rcp(pi);

    float bluenoise     = ditherBluenoise();

    if (water){
        vec3 flatNormal     = normalize(cross(dFdx(scenePos0), dFdy(scenePos0)));
        vec3 flatViewNormal = normalize(mat3(gbufferModelView) * flatNormal);

        vec3 normalCorrected = dot(viewNormal, normalize(viewPos1)) > 0.0 ? -viewNormal : viewNormal;

        vec3 refractedDir   = refract2(normalize(viewPos1), normalCorrected, flatViewNormal, rcp(1.33));
        //vec3 refractedDir   = refract(normalize(viewPos1), normalCorrected, rcp(1.33));

        float refractedDist = distance(viewPos0, viewPos1);

        vec3 refractedPos   = viewPos1 + refractedDir * refractedDist;

        vec3 screenPos      = viewToScreenSpace(refractedPos);

        float distToEdge    = max2(abs(screenPos.xy * 2.0 - 1.0));
            distToEdge      = sqr(sstep(distToEdge, 0.7, 1.0));

        screenPos.xy    = mix(screenPos.xy, coord, distToEdge);

        //vec2 refractionDelta = coord - screenPos.xy;

        float sceneDepth    = texture(depthtex1, screenPos.xy).x;

        if (sceneDepth > sceneDepth0) {
            sceneDepth1 = sceneDepth;
            viewPos1    = screenToViewSpace(vec3(screenPos.xy, sceneDepth1));
            scenePos1   = viewToSceneSpace(viewPos1);

            sceneColor.rgb  = texture(colortex0, screenPos.xy).rgb;
        }
    }

    float dist0     = distance(scenePos0, gbufferModelViewInverse[3].xyz);
    float dist1     = distance(scenePos1, gbufferModelViewInverse[3].xyz);

    vec3 skyboxColor = getSky(normalize(viewPos0));

    if (!landMask(sceneDepth1)) skyboxColor = mix(skyboxColor, sceneColor.rgb, sstep(-sceneDir1.y, -0.15, 0.0));

    #ifdef cloudVolumeEnabled
    vec4 clouds     = cloudSystem(normalize(scenePos1), scenePos1, landMask(sceneDepth1), dot(normalize(viewPos1), cloudLightDirView), ditherBluenoise(), skyboxColor);

    if (eyeAltitude < cloudVolume0Alt) sceneColor.rgb  = sceneColor.rgb * clouds.a + clouds.rgb;
    #endif

    if (translucent && isEyeInWater==0){
        if (!water) {
            #ifdef fogVolumeEnabled
                mat2x3 waterVol   = volumetricFog2(scenePos1, scenePos0, normalize(scenePos0), !landMask(sceneDepth0), vDotL, bluenoise, caveMult);
                applyFogData(sceneColor.rgb, waterVol);
            #endif
        }

        if (water) {
            sceneColor.rgb = waterFog(sceneColor.rgb, dist1-dist0, fogcol);
        }
        sceneColor.rgb  = blendTranslucencies(sceneColor.rgb, tex3, translucentAlbedo);
    }

    /* ------ reflections ------ */
    
    #ifdef resourcepackReflectionsEnabled

    if (water || material.roughness < 0.95){
        vec3 viewDir    = normalize(viewPos0);

        vec4 reflection = vec4(0.0);
        float skyOcclusion  = cubeSmooth(sqr(linStep(sceneLmap.y, roughReflectionsThreshold - 0.2, roughReflectionsThreshold)));

        vec3 metalAlbedo = translucentAlbedo;

        vec3 fresnel    = vec3(0.0);

        #ifdef roughReflectionsEnabled

        if (material.roughness < 0.0002 || water) {
            vec3 reflectDir = reflect(viewDir, viewNormal);

            if (dot(viewDir, viewNormal) > 0.0) viewNormal = -viewNormal;

            vec3 viewNormalSSR  = viewNormal;

            if (dot(viewDir, viewNormalSSR) > 0.0) viewNormalSSR = -viewNormalSSR;

            reflection          = screenRT(viewPos0, viewNormalSSR, bluenoise, translucent);

            if (reflection.a < 0.5 && skyOcclusion > 0.0) {
                vec2 sphereCoord    = unprojectSphere(mat3(gbufferModelViewInverse) * reflectDir);
                    sphereCoord    *= rcp(skyCaptureResolution);
                    //sphereCoord.y  += exp2(-SKY_RENDER_LOD) * 2.0;

                reflection.rgb      = texture(colortex5, sphereCoord).rgb;
                reflection.a        = 1.0;
                reflection         *= skyOcclusion;
            }

            fresnel   = BRDFfresnel(-viewDir, viewNormal, material, metalAlbedo);
        } else {
            float ssrAlpha  = 1.0 - sstep(material.roughness, roughSSR_maxR - 0.05, roughSSR_maxR);
            
            mat3 rot    = getRotationMat(vec3(0, 0, 1), viewNormal);
            vec3 tangentV = viewDir * rot;
            float noise = bluenoise;
            float dither = ditherGradNoiseTemporal();

            const uint steps    = roughReflectionSamples;
            const float rSteps  = 1.0 / float(steps);

            for (uint i = 0; i < steps; ++i) {
                vec2 xy         = vec2(fract((i + noise) * sqr(32.0) * phi), (i + noise) * rSteps);
                vec3 roughNrm   = rot * ggxFacetDist(-tangentV, material.roughness, xy);

                if (dot(viewDir, roughNrm) > 0.0) roughNrm = -roughNrm;

                vec3 reflectDir = reflect(viewDir, roughNrm);

                vec3 viewNormalSSR  = roughNrm;

                if (dot(viewDir, viewNormalSSR) > 0.0) viewNormalSSR = -viewNormalSSR;

                vec4 currReflection     = vec4(0.0);

                if (ssrAlpha > 0.0) currReflection = screenRT_Fast(viewPos0, viewNormalSSR, dither, translucent);
                
                if (currReflection.a < 0.5 && skyOcclusion > 0.0) {
                    vec3 worldDir       = mat3(gbufferModelViewInverse) * reflectDir;
                    vec2 sphereCoord    = unprojectSphere(worldDir);
                        sphereCoord    *= rcp(skyCaptureResolution);
                        //sphereCoord.y  += exp2(-SKY_RENDER_LOD) * 2.0;

                    currReflection.rgb  = texture(colortex5, sphereCoord).rgb;
                    currReflection.a    = 1.0;
                    currReflection     *= expf(-max0(-worldDir.y * pi)) * 0.66 + 0.34;
                    currReflection     *= skyOcclusion;
                }
                reflection += currReflection;
                fresnel    += BRDFfresnel(-viewDir, roughNrm, material, metalAlbedo);
            }
            reflection *= rSteps;
            fresnel    *= rSteps;
            reflection.a *= 1.0 - linStep(material.roughness, roughReflectionsThreshold * 0.66, roughReflectionsThreshold);
        }

        #else

            vec3 reflectDir = reflect(viewDir, viewNormal);

            vec3 viewNormalSSR  = viewNormal;

            if (dot(viewDir, viewNormalSSR) > 0.0) viewNormalSSR = -viewNormalSSR;

            reflection          = screenRT(viewPos0, viewNormalSSR, bluenoise, translucent);

            if (reflection.a < 0.5 && skyOcclusion > 0.0) {
                vec2 sphereCoord    = unprojectSphere(mat3(gbufferModelViewInverse) * reflectDir);
                    sphereCoord    *= rcp(skyCaptureResolution);
                    //sphereCoord.y  += exp2(-SKY_RENDER_LOD) * 2.0;

                reflection.rgb      = texture(colortex5, sphereCoord).rgb;
                reflection.a        = 1.0;
                reflection         *= skyOcclusion;
            }

            reflection.a *= 1.0 - linStep(material.roughness, roughReflectionsThreshold * 0.66, roughReflectionsThreshold);

            fresnel   = BRDFfresnel(-viewDir, viewNormal, material, metalAlbedo);

        #endif
        
        //vec3 fresnel   = BRDFfresnel(-viewDir, viewNormal, material, metalAlbedo);

        if (material.conductor && material.conductorComplex) reflection.rgb *= sqrt(metalAlbedo);
        
        if (material.conductor) {
            float diffuseLum    = getLuma(sceneColor.rgb);
            float reflectionLum = getLuma(reflection.rgb * fresnel);

            float darknessComp  = saturate(-(reflectionLum - diffuseLum / pi) / max(diffuseLum, reflectionLum));
                darknessComp   *= 0.25;

            sceneColor.rgb = mix(sceneColor.rgb, reflection.rgb * fresnel + sceneColor.rgb * darknessComp, reflection.a);
        }
        else sceneColor.rgb = mix(sceneColor.rgb, reflection.rgb, vec3(reflection.a) * fresnel);

        if (water || material.conductor) {
            vec3 shadowPos  = getShadowmapPos(scenePos0, 0.08);
            float shadow    = texture(shadowtex1, shadowPos);

            #ifdef lightleakWorkaroundToggle
                shadow     *= sstep(sceneLmap.y, 0.1, 0.2);
            #endif

            //if (material.conductor) shadow *= auxMat.x;

            if (shadow > 1e-2) {
                vec3 directCol  = (sunAngle<0.5 ? lightColor[0] : lightColor[2] * 0.1) * lightFlip;
                if (water) sceneColor.rgb += BRDF_Beckmann(-viewDir, lightvecView, viewNormal, material) * directCol;
                //else sceneColor.rgb += BRDF(-viewDir, lightvecView, viewNormal, material, metalAlbedo) * directCol;
            }
        }

        #if DEBUG_VIEW == 6
            sceneColor.rgb = reflection.rgb;
        #endif
    }

    #else

    if (water || (wetness > 1e-2 && landMask(sceneDepth1)) || matID == 103){
        vec3 viewDir    = normalize(viewPos0);

        vec4 reflection = vec4(0.0);
        float skyOcclusion  = cubeSmooth(sqr(linStep(sceneLmap.y, 0.7, 0.9)));
        vec3 reflectDir = reflect(viewDir, viewNormal);

        if (dot(viewDir, viewNormal) > 0.0) viewNormal = -viewNormal;

        reflection          = screenRT(viewPos0, viewNormal, bluenoise, translucent);

        if (reflection.a < 0.5) {
            vec2 sphereCoord    = unprojectSphere(mat3(gbufferModelViewInverse) * reflectDir);
                sphereCoord    *= rcp(skyCaptureResolution);
                //sphereCoord.y  += exp2(-SKY_RENDER_LOD) * 2.0;

            reflection.rgb      = texture(colortex5, sphereCoord).rgb;
            reflection.a        = 1.0 * skyOcclusion;
        }
        
        vec3 fresnel   = BRDFfresnel(-viewDir, viewNormal, material);
        if (!(water || matID == 103)) fresnel *= sqr(wetness);
        
        sceneColor.rgb = mix(sceneColor.rgb, reflection.rgb, vec3(reflection.a) * fresnel);

        if (water || matID == 103) {
            vec3 shadowPos  = getShadowmapPos(scenePos0, 0.08);
            float shadow    = texture(shadowtex1, shadowPos);

            #ifdef lightleakWorkaroundToggle
                shadow     *= sstep(sceneLmap.y, 0.1, 0.2);
            #endif

            if (shadow > 1e-2) {
                vec3 directCol  = (sunAngle<0.5 ? lightColor[0] : lightColor[2] * 0.1) * lightFlip;
                sceneColor.rgb += BRDF_Beckmann(-viewDir, lightvecView, viewNormal, material) * directCol;
            }
        }

        #if DEBUG_VIEW == 6
            sceneColor.rgb = reflection.rgb;
        #endif
    }

    #endif

    if (isEyeInWater==1) {
        if (water) {
            #ifdef fogVolumeEnabled
                mat2x3 waterVol   = volumetricFog2(scenePos1, scenePos0, normalize(scenePos0), !landMask(sceneDepth0), vDotL, bluenoise, caveMult);
                applyFogData(sceneColor.rgb, waterVol);
            #endif
        }
        sceneColor.rgb  = blendTranslucencies(sceneColor.rgb, tex3, translucentAlbedo);
        
        sceneColor.rgb = waterFog(sceneColor.rgb, dist0, fogcol);
    }

    if (isEyeInWater == 2) {
        sceneColor.rgb  = blendTranslucencies(sceneColor.rgb, tex3, translucentAlbedo);
        sceneColor.rgb = lavaFog(sceneColor.rgb, dist0);
    }
    
    if (isEyeInWater==0) {
        #ifdef fogVolumeEnabled

        mat2x3 fogData  = volumetricFog(scenePos0, normalize(scenePos0), !landMask(sceneDepth0), vDotL, bluenoise, caveMult);

        if (landMask(sceneDepth0)) sceneColor.rgb  = chunkEdgeFog(sceneColor.rgb, sceneDir0.y, skyboxColor, dist0);

        applyFogData(sceneColor.rgb, fogData);
        #else
        if (landMask(sceneDepth0)) sceneColor.rgb  = chunkEdgeFog(sceneColor.rgb, sceneDir0.y, skyboxColor, dist0);
        #endif
    }

    //sceneColor.rgb = skyboxColor;

    #ifdef cloudVolumeEnabled
    if (eyeAltitude > cloudVolume0Alt) sceneColor.rgb  = sceneColor.rgb * clouds.a + clouds.rgb;
    #endif

    scene   = makeDrawbuffer(sceneColor);
}